%---------------------------------------------------------------
%
% Addition of mechanical structure to the crystal cavity with high Q-factor
% calculation of the mechanical bandstructure
%
% the data will be saved in the file as given in the function definition
%
%---------------------------------------------------------------


function opt_best_unit_cell(geom, Eval_opt)

    format long
    clear all;
    close all;
   
    tic;
    import com.comsol.model.*
    import com.comsol.model.util.*

    model = ModelUtil.create('Model');
    model.modelNode.create('comp1');
    geometry = model.geom.create('geom1',3);

    disp('Setting Paramters... ');
        
%%   parameters and definitions
    
    c = 299792458;

    % Parameters of the snowflake (slightly modified from Snowflake_topological_insulator paper)
    geom.mech.d = 220e-9;
    geom.mech.r = 180e-9;
    geom.mech.delta_r = 0; % Shift the Dirac cone in kx direction
    geom.mech.r1 = geom.mech.r + geom.mech.delta_r;
    geom.mech.r2 = (3*geom.mech.r - geom.mech.r1)/2;
    geom.mech.r3 = (3*geom.mech.r - geom.mech.r1)/2;
    geom.mech.w = 73e-9;
    geom.mech.a = 500e-9;
    geom.mech.rad_curv = 20e-9;
    geom.mech.rad_curv2 = 12e-9;
    geom.mech.resolution = 40; % Resolution of the snowflake polygon

    % Minimum distance between optical cavity hole and the snowflake hole
    geom.bridge_width = 70e-9;

    % Parameters of the optical cavity
    geom.opt.scale = 1.077; % Do not change this scaling parameter for CAD sweeps; Change scale_structure and scale_radius
    geom.opt.delta_a = -9e-9*geom.opt.scale;
    geom.opt.a1 = 425e-9*geom.opt.scale;
    geom.opt.a2 = geom.opt.a1+geom.opt.delta_a;
    geom.opt.a3 = geom.opt.a1+2*geom.opt.delta_a;
    geom.opt.a4 = geom.opt.a1+3*geom.opt.delta_a;
    geom.opt.r = 130e-9*geom.opt.scale;
    geom.opt.W = 0;
    geom.opt.h = 130e-9*geom.opt.scale;
    geom.opt.b = 150e-9*geom.opt.scale;

    geom.opt.nh = 20;
    geom.opt.nv = 40;
    geom.opt.defect_holes = 2*geom.opt.nv;
    geom.opt.defect_width = geom.opt.r*2+30e-9;
    geom.opt.defect_length = geom.opt.a1*100;
    geom.opt.cavity_rot = 0;
    geom.opt.cavity_shift = 4; % Cavity shift from the center of triangle
    
%     geom.opt.scale_top_gap = 1;
    geom.opt.scale_top_gap = 0.783333;

    % Determining the scale factor of the snowflake so that optical cavity fits
    % inside the structure
    n = linspace(1,20,20);
    s = ((n+1)*geom.opt.a1 + 2*geom.opt.a2 + 2*geom.opt.a3 + geom.opt.a4)*sqrt(3)/geom.mech.a;
    index = 14; % 13 bad, 14 good
    num = n(index);
    geom.mech.scale = s(index);

    % Scaling the parameters of the snowflake by the above scale factor
    geom.mech.amech = geom.mech.a*geom.mech.scale;
    geom.mech.r = geom.mech.r*geom.mech.scale;
    geom.mech.r1 = geom.mech.r1*geom.mech.scale;
    geom.mech.r2 = geom.mech.r2*geom.mech.scale;
    geom.mech.r3 = geom.mech.r3*geom.mech.scale;
    geom.mech.w = geom.mech.w*geom.mech.scale;
    geom.mech.a = geom.mech.a*geom.mech.scale;
    geom.mech.rad_curv = geom.mech.rad_curv*geom.mech.scale;
    geom.mech.rad_curv2 = geom.mech.rad_curv2*geom.mech.scale;

    % Height of the snowflake triangle (not the triangle where topological edge
    % state is travelling)
    geom.mech.gap_defect_length = geom.mech.amech*sqrt(3)/2 - geom.mech.w*3/2;

    % Mesh parameters
    Eval_opt.hmax_air = 600e-9;
    Eval_opt.hmax = 90e-9;
    Eval_opt.hmin = 25e-9;
    
    Eval_opt.lambda = 1.5e-6; % Bounding box height = 2*lambda
    Eval_opt.qual_limit = 1e4; % Get the optical modes with Q greater than qual_limit
    
    % Mirror symmetry about xy plane
    Eval_opt.bc = 'TE';
    
    % .mph and .mat output files
    Eval_opt.filename = 'opt_best_unit_cell';
    
    % Optical mode at gamma point
    Eval_opt.kx = 0;
    Eval_opt.ky = 0;
    
    % Comsol output parameters
    Eval_opt.NumberOfFrequencies = 15;
    Eval_opt.CentralFrequency = 198.52e12;
    Eval_opt.freq = zeros(Eval_opt.NumberOfFrequencies*2,1);
    Eval_opt.damp = zeros(Eval_opt.NumberOfFrequencies*2,1);
    
%%   COMSOL-GLOBAL PARAMETERS
  
    model.param.set('d', num2str(geom.mech.d));
    model.param.set('r', num2str(geom.mech.r));
    model.param.set('w', num2str(geom.mech.w));
    model.param.set('a', num2str(geom.mech.a));
    model.param.set('a1', num2str(geom.opt.a1));
    model.param.set('a2', num2str(geom.opt.a2));
    model.param.set('a3', num2str(geom.opt.a3));
    model.param.set('a4', num2str(geom.opt.a4));
    model.param.set('r', num2str(geom.opt.r));
    model.param.set('W', num2str(geom.opt.W));
    model.param.set('height', num2str(geom.opt.h));
    model.param.set('b', num2str(geom.opt.b));
    model.param.set('hmax', num2str(Eval_opt.hmax));
    model.param.set('hmin', num2str(Eval_opt.hmin));
    model.param.set('lambda', num2str(Eval_opt.lambda));
    model.param.set('hmax_air', num2str(Eval_opt.hmax_air));

    mphsave(model, Eval_opt.filename);

%%   COMSOL-GEOMETRY
    disp('Building Geometry... ');
    
    workplane = geometry.feature.create('wp1','WorkPlane');
    
    % Hexagon of the small unit cell
    hex_x = [0 0 geom.mech.amech/2 geom.mech.amech geom.mech.amech geom.mech.amech/2 0];
    hex_y = [-geom.mech.amech/(2*sqrt(3)) geom.mech.amech/(2*sqrt(3)) geom.mech.amech/sqrt(3) geom.mech.amech/(2*sqrt(3)) -geom.mech.amech/(2*sqrt(3)) -geom.mech.amech/sqrt(3) -geom.mech.amech/(2*sqrt(3))];
    
    kite_x = [0 geom.mech.amech/2 geom.mech.amech geom.mech.amech/2 0];
    kite_y = geom.mech.amech*sqrt(3)/2 + [0 geom.mech.amech*sqrt(3)/2 0 -geom.mech.amech*sqrt(3)/2 0];

    % Rectangular area in the unit cell where the geometry has to be specified
    rect1_x = [geom.mech.amech/4 geom.mech.amech/4 geom.mech.amech*3/4 geom.mech.amech*3/4 geom.mech.amech/4];
    rect1_y = [0 geom.mech.amech*sqrt(3)/2 geom.mech.amech*sqrt(3)/2 0 0];

    rect2_x = [geom.mech.amech*3/4 geom.mech.amech*3/4 geom.mech.amech geom.mech.amech geom.mech.amech*3/4];
    rect2_y = [-geom.mech.amech*sqrt(3)/2 geom.mech.amech*sqrt(3)/2 geom.mech.amech*sqrt(3)/2 -geom.mech.amech*sqrt(3)/2 -geom.mech.amech*sqrt(3)/2];
    
    % Draw snowflakes
    % angle = -60;
    x_cen = geom.mech.amech/2;
    y_cen = 0;
    [snowflake_cen_x, snowflake_cen_y] = coordinates_snowflake(x_cen, y_cen, geom.mech.r, geom.mech.w);

    mech = workplane.geom.create('pol2', 'Polygon');
    mech.set('source', 'table');
    for pol_i=1:length(snowflake_cen_x(1:end-1))
        mech.setIndex('table', num2str(snowflake_cen_x(pol_i),'%10.9e'), pol_i-1, 0);  %value,row, coloumn
        mech.setIndex('table', num2str(snowflake_cen_y(pol_i),'%10.9e'), pol_i-1, 1);
    end
    
    % Fillet the snowflake corners
    epsilon = 100e-9;
    disksel = workplane.geom.create('disksel1', 'DiskSelection');
    disksel.set('entitydim', 0);
    disksel.set('posx', num2str(geom.mech.amech/2,'%10.9e'));
    disksel.set('posy', '0');
    disksel.set('r', num2str(geom.mech.w + epsilon,'%10.9e'));
    disksel.set('condition', 'inside');
    
    disksel = workplane.geom.create('disksel2', 'DiskSelection');
    disksel.set('entitydim', 0);
    disksel.set('posx', num2str(geom.mech.amech/2,'%10.9e'));
    disksel.set('posy', '0');
    disksel.set('r', num2str(sqrt(geom.mech.r^2 + (geom.mech.w/2)^2) + epsilon,'%10.9e'));
    disksel.set('rin', num2str(geom.mech.w + epsilon,'%10.9e'));
    disksel.set('condition', 'inside');
    
    fil = workplane.geom.create('fil1', 'Fillet');
    fil.set('radius', num2str(geom.mech.rad_curv2,'%10.9e'));
    fil.selection('point').named('disksel1');
    
    fil = workplane.geom.create('fil2', 'Fillet');
    fil.set('radius', num2str(geom.mech.rad_curv,'%10.9e'));
    fil.selection('point').named('disksel2');
    
    [snowflake_cen_x, snowflake_cen_y] = coordinates_round_snowflake_comsol(x_cen, y_cen, geom.mech.r, geom.mech.w, geom.mech.rad_curv, geom.mech.rad_curv2, geom.mech.resolution);
    
    snowflake_x4 = snowflake_cen_x + geom.mech.amech/2;
    snowflake_y4 = snowflake_cen_y + geom.mech.amech*sqrt(3)/2;
    
    % Position of gap defect
%     x_cen = geom.mech.amech/2;
%     y_cen = -geom.mech.amech/sqrt(3);
%     geom.mech.gap_defect_x = [x_cen-geom.mech.gap_defect_width/2 x_cen-geom.mech.gap_defect_width/2 x_cen+geom.mech.gap_defect_width/2 x_cen+geom.mech.gap_defect_width/2];
%     geom.mech.gap_defect_y = [y_cen-geom.mech.gap_defect_length/3 y_cen+geom.mech.gap_defect_length*2/3 y_cen+geom.mech.gap_defect_length*2/3 y_cen-geom.mech.gap_defect_length/3];
    
    % Coordinates of triangle where the hole sizes are scaled in order to create topological bandgap
    x_cen = geom.mech.amech;
    y_cen = geom.mech.amech/(2*sqrt(3));
    geom.mech.tri_defect_x = [x_cen-geom.mech.gap_defect_length/sqrt(3) x_cen x_cen+geom.mech.gap_defect_length/sqrt(3)];
    geom.mech.tri_defect_y = [y_cen-geom.mech.gap_defect_length/3 y_cen+geom.mech.gap_defect_length*2/3 y_cen-geom.mech.gap_defect_length/3];
    
    % Position of optical cavity
    x_cen = geom.mech.amech/2;
    y_cen = geom.mech.amech/sqrt(3) + geom.opt.cavity_shift*geom.opt.a1/2;
    temp_x = [x_cen-geom.opt.defect_width/2 x_cen-geom.opt.defect_width/2 x_cen+geom.opt.defect_width/2 x_cen+geom.opt.defect_width/2];
    temp_y = [y_cen-geom.opt.defect_length/2 y_cen+geom.opt.defect_length/2 y_cen+geom.opt.defect_length/2 y_cen-geom.opt.defect_length/2];
    [geom.opt.defect_x, geom.opt.defect_y] = rotate(temp_x, temp_y, x_cen, y_cen, geom.opt.cavity_rot);

    % plot(geom.opt.defect_x, geom.opt.defect_y, 'color', 'red', 'LineWidth', 0.5)

    delta2 = [geom.opt.a1*ones(1,geom.opt.nv-5), ...
        (geom.opt.a1+geom.opt.a2)/2, ... %1
        geom.opt.a2, ... %2
        (geom.opt.a2+geom.opt.a3)/2, ... %3
        geom.opt.a3, ... %4
        (geom.opt.a3+geom.opt.a4)/2, ... %5
        geom.opt.a4, ... %6
        (geom.opt.a3+geom.opt.a4)/2, ... %7
        geom.opt.a3, ... %8
        (geom.opt.a2+geom.opt.a3)/2, ... %9
        geom.opt.a2, ... %10
        (geom.opt.a1+geom.opt.a2)/2, ... %11
        geom.opt.a1*ones(1,geom.opt.nv-5)];

    delta1 = [geom.opt.a1*ones(1,geom.opt.nv-6), ...
        (geom.opt.a1+(geom.opt.a1+geom.opt.a2)/2)/2, ... %1
        (geom.opt.a2+(geom.opt.a1+geom.opt.a2)/2)/2, ... %2
        (geom.opt.a2+(geom.opt.a2+geom.opt.a3)/2)/2, ... %3
        (geom.opt.a3+(geom.opt.a2+geom.opt.a3)/2)/2, ... %4
        (geom.opt.a3+(geom.opt.a3+geom.opt.a4)/2)/2, ... %5
        (geom.opt.a4+(geom.opt.a3+geom.opt.a4)/2)/2, ... %6
        (geom.opt.a4+(geom.opt.a3+geom.opt.a4)/2)/2, ... %7
        (geom.opt.a3+(geom.opt.a3+geom.opt.a4)/2)/2, ... %8
        (geom.opt.a3+(geom.opt.a2+geom.opt.a3)/2)/2, ... %9
        (geom.opt.a2+(geom.opt.a2+geom.opt.a3)/2)/2, ... %10
        (geom.opt.a2+(geom.opt.a1+geom.opt.a2)/2)/2, ... %11
        (geom.opt.a1+(geom.opt.a1+geom.opt.a2)/2)/2, ... %12
        geom.opt.a1*ones(1,geom.opt.nv-6)];
    x = zeros(2*geom.opt.nv+1, 2*geom.opt.nh+1);
    y = zeros(2*geom.opt.nv+1, 2*geom.opt.nh+1);
    rad = zeros(2*geom.opt.nv+1, 2*geom.opt.nh+1);
    radb = zeros(2*geom.opt.nv+1, 2*geom.opt.nh+1);

    for j = -geom.opt.nh:geom.opt.nh
        if j==-geom.opt.nh
            if rem(geom.opt.nv,2) == 0
                x(:,j+geom.opt.nh+1) = -geom.opt.nh*geom.opt.a1*sqrt(3)/2 - geom.opt.W/2;
            else
                x(:,j+geom.opt.nh+1) = -(geom.opt.nh+1)*geom.opt.a1*sqrt(3)/2 - geom.opt.W/2;
            end
        else
            if j==0 || j==1
                x(:,j+geom.opt.nh+1) = x(:,j+geom.opt.nh) + (geom.opt.a1*sqrt(3) + geom.opt.W)/2;
            else
                x(:,j+geom.opt.nh+1) = x(:,j+geom.opt.nh) + geom.opt.a1*sqrt(3)/2;
            end
        end

        for i=-geom.opt.nv:geom.opt.nv
            rad(i+geom.opt.nv+1,j+geom.opt.nh+1) = geom.opt.r;

            if i==-geom.opt.nv
                if rem(j,2)==0
                    y(i+geom.opt.nv+1,j+geom.opt.nh+1) = -sum(delta1(1:geom.opt.nv));
                else
                    y(i+geom.opt.nv+1,j+geom.opt.nh+1) = -sum(delta2(1:geom.opt.nv))-geom.opt.a3/2;
                end
            else
                if rem(j,2)==0
                    y(i+geom.opt.nv+1,j+geom.opt.nh+1) = y(i+geom.opt.nv,j+geom.opt.nh+1)+delta1(i+geom.opt.nv);
                else
                    y(i+geom.opt.nv+1,j+geom.opt.nh+1) = y(i+geom.opt.nv,j+geom.opt.nh+1)+delta2(i+geom.opt.nv);
                end
            end
        end
    end

    temp_x = x+x_cen;
    temp_y = y+y_cen;
    [x, y] = rotate(temp_x, temp_y, x_cen, y_cen, geom.opt.cavity_rot);

    % Draw cavity holes in region 1 of geometry schematic
    count = 1;
    count1 = 1;
    for j = -geom.opt.nh:geom.opt.nh
        for i = -geom.opt.nv:geom.opt.nv
            x1 = x(i+geom.opt.nv+1,j+geom.opt.nh+1);
            y1 = y(i+geom.opt.nv+1,j+geom.opt.nh+1);

            r1 = rad(i+geom.opt.nv+1,j+geom.opt.nh+1);
            rh = geom.opt.h;
            rb = geom.opt.b;

            if inpolygon(x1, y1, geom.opt.defect_x, geom.opt.defect_y) == 0
                phi = linspace(0,2*pi,37);
                xx = x1 + r1*cos(phi);
                yy = y1 + r1*sin(phi);
            else
                y1 = y(i+geom.opt.nv+1,j+1+geom.opt.nh+1);
                phi = linspace(0,2*pi,37);
                xx = x1 + rb*cos(phi);
                yy = y1 + rh*sin(phi);
            end

            for k=1:length(xx)-1
                check(k)=abs(p_poly_dist1(xx(k),yy(k),snowflake_cen_x,snowflake_cen_y));
            end
            
            in_scale_tri = inpolygon(xx(1:end-1), yy(1:end-1), geom.mech.tri_defect_x, geom.mech.tri_defect_y);
            in_scale_tri2 = inpolygon(xx(1:end-1), yy(1:end-1), geom.mech.tri_defect_x-geom.mech.amech, geom.mech.tri_defect_y);
            in_defect = inpolygon(xx(1:end-1), yy(1:end-1), geom.opt.defect_x, geom.opt.defect_y);
            in_rect1 = inpolygon(xx(1:end-1), yy(1:end-1), rect1_x, rect1_y);
            in_uc = inpolygon(xx(1:end-1), yy(1:end-1), hex_x, hex_y);
            in = inpolygon(xx(1:end-1), yy(1:end-1), snowflake_cen_x, snowflake_cen_y);

            dist1 = x1 - geom.mech.amech/4;
            dist2 = geom.mech.amech*3/4 - x1;

            if min(check)>geom.bridge_width && sum(in_uc)==length(xx)-1 && sum(in_rect1)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist1>geom.opt.a1*sqrt(3)/2 && dist2>geom.opt.a1*sqrt(3)/2
                if sum(in_scale_tri) == length(xx)-1 || sum(in_scale_tri2) == length(xx)-1
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp(count) = {strcat('c','x',num2str(i),'y',num2str(j))};
                count = count+1;
                
            elseif min(check)>geom.bridge_width && sum(in_uc)>0 && sum(in_rect1)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist1>geom.opt.a1*sqrt(3)/2 && dist2>geom.opt.a1*sqrt(3)/2
                if sum(in_scale_tri) == length(xx)-1 || sum(in_scale_tri2) == length(xx)-1
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp1(count1) = {strcat('c','x',num2str(i),'y',num2str(j))};
                count1 = count1+1;

            elseif min(check)>geom.bridge_width && sum(in_uc)>0 && sum(in_rect1)==length(xx)-1 && sum(in)==0 && sum(in_defect)~=0 && geom.opt.h ~=0 && geom.opt.b ~=0 && dist1>geom.opt.a1*sqrt(3)/2 && dist2>geom.opt.a1*sqrt(3)/2
                if sum(in_scale_tri) == length(xx)-1 || sum(in_scale_tri2) == length(xx)-1
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb*geom.opt.scale_top_gap,'%10.9e') num2str(rh*geom.opt.scale_top_gap,'%10.9e')});
                else
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb,'%10.9e') num2str(rh,'%10.9e')});
                end
                
                inp(count) = {strcat('c','x',num2str(i),'y',num2str(j))};
                count = count+1;
                
            elseif min(check)>geom.bridge_width && sum(in_uc)>0 && sum(in_rect1)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist2<geom.opt.a1*sqrt(3)/2
                x1 = x1 + (dist2 - geom.opt.a1*sqrt(3)/2)/2;
                
                if sum(in_scale_tri) == length(xx)-1 || sum(in_scale_tri2) == length(xx)-1
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                if sum(in_uc)==length(xx)-1
                    inp(count) = {strcat('c','x',num2str(i),'y',num2str(j))};
                    count = count+1;
                else
                    inp1(count1) = {strcat('c','x',num2str(i),'y',num2str(j))};
                    count1 = count1+1;
                end
                
            elseif min(check)>geom.bridge_width && sum(in_uc)>0 && sum(in_rect1)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist1<geom.opt.a1*sqrt(3)/2
                x1 = x1 + (geom.opt.a1*sqrt(3)/2 - dist1)/2;
                
                if sum(in_scale_tri) == length(xx)-1 || sum(in_scale_tri2) == length(xx)-1
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                if sum(in_uc)==length(xx)-1
                    inp(count) = {strcat('c','x',num2str(i),'y',num2str(j))};
                    count = count+1;
                else
                    inp1(count1) = {strcat('c','x',num2str(i),'y',num2str(j))};
                    count1 = count1+1;
                end
            end
        end
    end
    
    uni = workplane.geom.create('uni_region1', 'Union');
    uni.selection('input').set(inp);
    
    uni = workplane.geom.create('uni_region1_bdd', 'Union');
    uni.selection('input').set(inp1);
    
    copy = workplane.geom.create('cp_region1_bdd_1', 'Copy');
    copy.selection('input').set({'uni_region1_bdd'});
    copy.set('displx', num2str(geom.mech.amech/2,'%10.9e'));
    copy.set('disply', num2str(-geom.mech.amech*sqrt(3)/2,'%10.9e'));
    
    copy = workplane.geom.create('cp_region1_bdd_2', 'Copy');
    copy.selection('input').set({'uni_region1_bdd'});
    copy.set('displx', num2str(-geom.mech.amech/2,'%10.9e'));
    copy.set('disply', num2str(-geom.mech.amech*sqrt(3)/2,'%10.9e'));
    
    % Constructing cavity holes in region 2 of geometry schematic
    geom.opt.defect_x = geom.opt.defect_x + geom.mech.amech/2;
    geom.opt.defect_y = geom.opt.defect_y - geom.mech.amech*sqrt(3)/2;
    
%     geom.mech.gap_defect_x = geom.mech.gap_defect_x + geom.mech.amech/2;
%     geom.mech.gap_defect_y = geom.mech.gap_defect_y + geom.mech.amech*sqrt(3)/2;

    clear inp inp1;
    count = 1;
    count1 = 1;
    for j = -geom.opt.nh:geom.opt.nh
        for i = -geom.opt.nv:geom.opt.nv
            x1 = x(i+geom.opt.nv+1,j+geom.opt.nh+1) + geom.mech.amech/2;
            y1 = y(i+geom.opt.nv+1,j+geom.opt.nh+1) - geom.mech.amech*sqrt(3)/2;

            r1 = rad(i+geom.opt.nv+1,j+geom.opt.nh+1);
            rh = geom.opt.h;
            rb = geom.opt.b;

            if inpolygon(x1, y1, geom.opt.defect_x, geom.opt.defect_y) == 0
                phi = linspace(0,2*pi,37);
                xx = x1 + r1*cos(phi);
                yy = y1 + r1*sin(phi);
            else
                y1 = y(i+geom.opt.nv+1,j+1+geom.opt.nh+1) - geom.mech.amech*sqrt(3)/2;
                phi = linspace(0,2*pi,37);
                xx = x1 + rb*cos(phi);
                yy = y1 + rh*sin(phi);
            end

            for k=1:length(xx)-1
                check(k)=abs(p_poly_dist1(xx(k),yy(k),snowflake_cen_x,snowflake_cen_y));
                check1(k)=abs(p_poly_dist1(xx(k),yy(k),snowflake_x4,snowflake_y4));
            end
            
            in_scale_tri = inpolygon(xx(1:end-1), yy(1:end-1), geom.mech.tri_defect_x, geom.mech.tri_defect_y);
            in_defect = inpolygon(xx(1:end-1), yy(1:end-1), geom.opt.defect_x, geom.opt.defect_y);
%             in_gap_defect = inpolygon(xx(1:end-1), yy(1:end-1), geom.mech.gap_defect_x, geom.mech.gap_defect_y);
            in_rect2 = inpolygon(xx(1:end-1), yy(1:end-1), rect2_x, rect2_y);
            in_uc = inpolygon(xx(1:end-1), yy(1:end-1), hex_x, hex_y);
            in = inpolygon(xx(1:end-1), yy(1:end-1), snowflake_cen_x, snowflake_cen_y);
            in1 = inpolygon(xx(1:end-1), yy(1:end-1), snowflake_x4, snowflake_y4);

            dist = x1 - geom.mech.amech*3/4;

%             if min(check)>geom.bridge_width && sum(in_gap_defect)~=0 && sum(in1)==0
%                 x1 = x1 - geom.mech.amech/2;
%                 y1 = y1 - geom.mech.amech*sqrt(3)/2;
%                 
%                 workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
%                 workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
%                 workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
% 
%                 inp1(count1) = {strcat('c2','x',num2str(i),'y',num2str(j))};
%                 count1 = count1+1;
            
            if min(check)>geom.bridge_width && sum(in_uc)==length(xx)-1 && sum(in_rect2)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist>geom.opt.a1*sqrt(3)/2
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count = count+1;

            elseif min(check)>geom.bridge_width && sum(in_rect2)>0 && sum(in)==0 && sum(in_uc)>0 && sum(in_defect)~=0 && geom.opt.h ~=0 && geom.opt.b ~=0 && dist>geom.opt.a1*sqrt(3)/2
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb*geom.opt.scale_top_gap,'%10.9e') num2str(rh*geom.opt.scale_top_gap,'%10.9e')});
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb,'%10.9e') num2str(rh,'%10.9e')});
                end

                inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count = count+1;

            elseif min(check)>geom.bridge_width && min(xx)<geom.mech.amech*3/4 && sum(in_uc)>0 && max(xx)>geom.mech.amech*3/4 && sum(in_rect2)>0 && sum(in)==0 && sum(in_defect)==0
                x1 = geom.mech.amech*3/4;
                
                if sum(in_uc)~=length(xx)-1
                    y1 = geom.mech.amech/sqrt(3) - abs((x1 - geom.mech.amech/2))/sqrt(3);
                    
                    x1 = x1 - geom.mech.amech/2;
                    y1 = y1 - geom.mech.amech*sqrt(3)/2;
                end

                phi = linspace(0,2*pi,37);
                xx = x1 + r1*cos(phi);
                yy = y1 + r1*sin(phi);

                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                if sum(in_uc)~=length(xx)-1
                    inp1(count1) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                    count1 = count1+1;
                else
                    inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                    count = count+1;
                end
                
            elseif min(check)>geom.bridge_width && sum(in_uc)==length(xx)-1 && sum(in_rect2)==length(xx)-1 && sum(in)==0 && sum(in_defect)==0 && dist<geom.opt.a1*sqrt(3)/2
                x1 = x1 + (geom.opt.a1*sqrt(3)/2 - dist)/2;
                
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count = count+1;
                
            elseif min(check1)>geom.bridge_width && sum(in_uc)==0 && sum(in_rect2)>0 && sum(in)==0 && sum(in_defect)==0 && sum(in1)==0 && min(xx)>geom.mech.amech*3/4 && min(yy)>0
                x1 = x1 - geom.mech.amech/2;
                y1 = y1 - geom.mech.amech*sqrt(3)/2;
                
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count = count+1;
                            
            elseif min(check1)>geom.bridge_width && sum(in_uc)==0 && sum(in_rect2)>0 && sum(in)==0 && sum(in_defect)~=0 && sum(in1)==0 && min(xx)>geom.mech.amech*3/4 && min(yy)>0
                x1 = x1 - geom.mech.amech/2;
                y1 = y1 - geom.mech.amech*sqrt(3)/2;
                
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb*geom.opt.scale_top_gap,'%10.9e') num2str(rh*geom.opt.scale_top_gap,'%10.9e')});
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)), 'Ellipse');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('semiaxes', {num2str(rb,'%10.9e') num2str(rh,'%10.9e')});
                end

                inp(count) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count = count+1;
                
            elseif min(check)>geom.bridge_width && sum(in_uc)>0 && sum(in_rect2)>0 && sum(in)==0 && sum(in1)==0 && min(xx)>geom.mech.amech*3/4 && min(yy)>0
                x1 = x1 - geom.mech.amech/2;
                y1 = y1 - geom.mech.amech*sqrt(3)/2;
                
                if sum(in_scale_tri) == length(xx)-1
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1*geom.opt.scale_top_gap,'%10.9e'));
                else
                    workplane.geom.create(strcat('c2','x',num2str(i),'y',num2str(j)),'Circle');
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('pos', {num2str(x1,'%10.9e') num2str(y1,'%10.9e')});
                    workplane.geom.feature(strcat('c2','x',num2str(i),'y',num2str(j))).set('r', num2str(r1,'%10.9e'));
                end

                inp1(count1) = {strcat('c2','x',num2str(i),'y',num2str(j))};
                count1 = count1+1;
            end
        end
    end
    
    uni = workplane.geom.create('uni_region2_bdd', 'Union');
    uni.selection('input').set(inp1);
    
    copy = workplane.geom.create('cp_region2_bdd', 'Copy');
    copy.selection('input').set({'uni_region2_bdd'});
    copy.set('displx', num2str(geom.mech.amech/2,'%10.9e'));
    copy.set('disply', num2str(geom.mech.amech*sqrt(3)/2,'%10.9e'));
    
    uni = workplane.geom.create('uni_region2', 'Union');
    uni.selection('input').set(inp);
    
    mir = workplane.geom.create('mir1', 'Mirror');
    mir.set('keep', true);
    mir.selection('input').set({'uni_region2' 'uni_region2_bdd' 'cp_region2_bdd'});
    mir.set('pos', {num2str(geom.mech.amech/2,'%10.9e') '0'});
    
    uni = workplane.geom.create('uni1', 'Union');
    uni.selection('input').set({'fil2' 'uni_region1' 'uni_region1_bdd' 'cp_region1_bdd_1' 'cp_region1_bdd_2' 'uni_region2' 'uni_region2_bdd' 'cp_region2_bdd' 'mir1'});
    
    copy = workplane.geom.create('copy1', 'Copy');
    copy.selection('input').set({'uni1'});
    copy.set('displx', num2str(geom.mech.amech/2,'%10.9e'));
    copy.set('disply', num2str(geom.mech.amech*sqrt(3)/2,'%10.9e'));
    
    copy = workplane.geom.create('copy2', 'Copy');
    copy.selection('input').set({'uni1'});
    copy.set('displx', num2str(-geom.mech.amech/2,'%10.9e'));
    copy.set('disply', num2str(geom.mech.amech*sqrt(3)/2,'%10.9e'));
    
    copy = workplane.geom.create('copy3', 'Copy');
    copy.selection('input').set({'uni1'});
    copy.set('disply', num2str(geom.mech.amech*sqrt(3),'%10.9e'));
    
    mech = workplane.geom.create('unit_cell_bdd', 'Polygon');
    mech.set('source', 'table');
    for i=1:length(kite_x(1:end-1))
        mech.setIndex('table', num2str(kite_x(i),'%10.9e'), i-1, 0);  %value,row, coloumn
        mech.setIndex('table', num2str(kite_y(i),'%10.9e'), i-1, 1);
    end
    
    workplane.geom.create('dif1', 'Difference');
    workplane.geom.feature('dif1').selection('input').set({'unit_cell_bdd'});
    workplane.geom.feature('dif1').selection('input2').set({'uni1' 'copy1' 'copy2' 'copy3'});
        
    ext = geometry.feature.create('ext1', 'Extrude');
    ext.set('workplane', 'wp1');
    ext.selection('input').set({'wp1'});
    ext.setIndex('distance', 'd/2', 0);
    
    workplane = geometry.feature.create('wp2','WorkPlane');
    
    mech = workplane.geom.create('unit_cell_bdd_air', 'Polygon');
    mech.set('source', 'table');
    for i=1:length(kite_x(1:end-1))
        mech.setIndex('table', num2str(kite_x(i),'%10.9e'), i-1, 0);  %value,row, coloumn
        mech.setIndex('table', num2str(kite_y(i),'%10.9e'), i-1, 1);
    end
    
    extair = geometry.feature.create('ext2', 'Extrude');
    extair.set('workplane', 'wp2');
    extair.selection('input').set({'wp2'});
    extair.setIndex('distance', '2*lambda', 0);
    
    geometry.run;
    figure(1)
    mphgeom(model);
    view(180,-90);
    mphsave(model, Eval_opt.filename)
    
%%   COMSOL-MATERIAL
    disp('Setting Material...');
    
    Silicon = model.material.create('mat2', 'Common', 'comp1');
    Silicon.name('Silicon');
    Silicon.propertyGroup('def').set('relpermeability', '1');
    Silicon.propertyGroup('def').set('electricconductivity', '1e-12[S/m]');
    Silicon.propertyGroup('def').set('relpermittivity', '11.7');

    model.material('mat2').label('Si - Silicon (single-crystal, anisotropic) 1');
    model.material('mat2').set('family', 'custom');
    model.material('mat2').set('lighting', 'cooktorrance');
    model.material('mat2').set('specular', 'custom');
    model.material('mat2').set('customspecular', [0.7843137254901961 1 1]);
    model.material('mat2').set('fresnel', 0.9);
    model.material('mat2').set('roughness', 0.1);
    model.material('mat2').set('shininess', 200);
    model.material('mat2').set('diffuse', 'custom');
    model.material('mat2').set('customdiffuse', [0.6666666666666666 0.6666666666666666 0.7058823529411765]);
    model.material('mat2').set('ambient', 'custom');
    model.material('mat2').set('customambient', [0.6666666666666666 0.6666666666666666 0.7058823529411765]);
    model.material('mat2').set('fresnel', 0.7);
    model.material('mat2').set('roughness', 0.5);
    model.material('mat2').propertyGroup('def').set('density', '2330[kg/m^3]');
    model.material('mat2').propertyGroup.create('Anisotropic', 'Anisotropic');
    model.material('mat2').propertyGroup('Anisotropic').set('D', {'166[GPa]' '64[GPa]' '166[GPa]' '64[GPa]' '64[GPa]' '166[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '80[GPa]'  ...
    '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '80[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]' '0[GPa]'  ...
    '80[GPa]'});
    
    epsilon = 0.01*min([geom.mech.amech, geom.mech.d/2, geom.opt.r]);
    coordBox = [-1, 1; -1, 1; -epsilon, geom.mech.d/2+epsilon];
    si_domain = mphselectbox(model,'geom1',coordBox,'domain');
    Silicon.selection.set(si_domain);
    
    Air = model.material.create('Air');
    Air.name('Air');
    Air.propertyGroup('def').set('relpermeability', '1');
    Air.propertyGroup('def').set('relpermittivity', '1');
    Air.propertyGroup('def').set('electricconductivity', '0[S/m]');
    
    coordBox = [-1, 1; -1, 1; -1, 1];
    all_domain = mphselectbox(model,'geom1',coordBox,'domain');
    air_domain = setdiff(all_domain, si_domain);
    Air.selection.set(air_domain);
   
    mphsave(model, Eval_opt.filename);
    
%%   COMSOL-MESH AND COMSOL-PHYSICS
    disp('Meshing and Setting Physics... ');
    
    mesh = model.mesh.create('mesh1', 'geom1');
    physics = model.physics.create('emw', 'ElectromagneticWaves', 'geom1');
    
    coordBox = [-1, 1; -1, 1; -epsilon, epsilon];
    si_b = mphselectbox(model,'geom1',coordBox,'boundary');
    if (Eval_opt.bc == 'TE')
        model.physics('emw').create('pmc1', 'PerfectMagneticConductor', 2);
        model.physics('emw').feature('pmc1').selection.set(si_b);
    elseif(Eval_opt.bc == 'TM')
        model.physics('emw').create('pec2', 'PerfectElectricConductor', 2);
        model.physics('emw').feature('pec2').selection.set(si_b);
    end
    
    rot = geometry.feature.create('rot1', 'Rotate');
    rot.set('rot', '60');
    rot.set('pos', {num2str(geom.mech.amech/2,'%10.9e') num2str(geom.mech.amech/sqrt(3),'%10.9e') '0'});
    rot.selection('input').set({'ext1' 'ext2'});

    coordBox = [-1, 1; geom.mech.amech/(2*sqrt(3))-epsilon, geom.mech.amech/(2*sqrt(3))+epsilon; -epsilon, geom.mech.d/2+epsilon];
    l1 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    coordBox = [-1, 1; geom.mech.amech/(2*sqrt(3))-epsilon, geom.mech.amech/(2*sqrt(3))+epsilon; -1, 1];
    temp = mphselectbox(model,'geom1',coordBox,'boundary');
    l1a = setdiff(temp,l1);
    
    coordBox = [-1, 1; geom.mech.amech/(2*sqrt(3))+geom.mech.amech*sqrt(3)/2-epsilon, geom.mech.amech/(2*sqrt(3))+geom.mech.amech*sqrt(3)/2+epsilon; -epsilon, geom.mech.d/2+epsilon];
    l2 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    coordBox = [-1, 1; geom.mech.amech/(2*sqrt(3))+geom.mech.amech*sqrt(3)/2-epsilon, geom.mech.amech/(2*sqrt(3))+geom.mech.amech*sqrt(3)/2+epsilon; -1, 1];
    temp = mphselectbox(model,'geom1',coordBox,'boundary');
    l2a = setdiff(temp,l2);
    
    freetri = mesh.create('ftri1', 'FreeTri');
    freetri.create('size1', 'Size');
    freetri.feature('size1').set('custom', 'on');
    freetri.feature('size1').set('hmax', 'hmax');
    freetri.feature('size1').set('hmin', 'hmin');
%     freetri.feature('size1').set('custom', false);
%     freetri.feature('size1').set('hauto', 3);
    freetri.selection.set([l1]);
    copy_face = mesh.create('cpf1', 'CopyFace');
    copy_face.selection('source').set([l1]);
    copy_face.selection('destination').set([l2]);
    
    freetri = mesh.create('ftri1a', 'FreeTri');
    freetri.create('size1', 'Size');
    freetri.feature('size1').set('custom', 'on');
    freetri.feature('size1').set('hmax', 'hmax_air');
    freetri.feature('size1').set('hmin', 'hmin');
%     freetri.feature('size1').set('custom', false);
%     freetri.feature('size1').set('hauto', 3);
    freetri.selection.set([l1a]);
    copy_face = mesh.create('cpf1a', 'CopyFace');
    copy_face.selection('source').set([l1a]);
    copy_face.selection('destination').set([l2a]);
    
    physics.feature.create('pc1', 'PeriodicCondition', 2);    % What does 2 stand for here?
    physics.feature('pc1').set('PeriodicType', 'Floquet');
    physics.feature('pc1').selection.set([l1 l1a l2 l2a]);
    
    rot = geometry.feature.create('rot2', 'Rotate');
    rot.set('rot', '-120');
    rot.set('pos', {num2str(geom.mech.amech/2,'%10.9e') num2str(geom.mech.amech/sqrt(3),'%10.9e') '0'});
    rot.selection('input').set({'rot1'});
    
    coordBox = [-1, 1; geom.mech.amech/(2*sqrt(3))-epsilon, geom.mech.amech/(2*sqrt(3))+epsilon; -epsilon, geom.mech.d/2+epsilon];
    m1 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    coordBox = [-1, 1; geom.mech.amech/(2*sqrt(3))-epsilon, geom.mech.amech/(2*sqrt(3))+epsilon; -1, 1];
    temp = mphselectbox(model,'geom1',coordBox,'boundary');
    m1a = setdiff(temp, m1);
    
    coordBox = [-1, 1; geom.mech.amech/(2*sqrt(3))+geom.mech.amech*sqrt(3)/2-epsilon, geom.mech.amech/(2*sqrt(3))+geom.mech.amech*sqrt(3)/2+epsilon; -epsilon, geom.mech.d/2+epsilon];
    m2 = mphselectbox(model,'geom1',coordBox,'boundary');
    
    coordBox = [-1, 1; geom.mech.amech/(2*sqrt(3))+geom.mech.amech*sqrt(3)/2-epsilon, geom.mech.amech/(2*sqrt(3))+geom.mech.amech*sqrt(3)/2+epsilon; -1, 1];
    temp = mphselectbox(model,'geom1',coordBox,'boundary');
    m2a = setdiff(temp, m2);
    
    freetri = mesh.create('ftri2', 'FreeTri');
    freetri.create('size1', 'Size');
    freetri.feature('size1').set('custom', 'on');
    freetri.feature('size1').set('hmax', 'hmax');
    freetri.feature('size1').set('hmin', 'hmin');
%     freetri.feature('size1').set('custom', false);
%     freetri.feature('size1').set('hauto', 3);
    freetri.selection.set([m2]);
    copy_face = mesh.create('cpf2', 'CopyFace');
    copy_face.selection('source').set([m2]);
    copy_face.selection('destination').set([m1]);
    
    freetri = mesh.create('ftri2a', 'FreeTri');
    freetri.create('size1', 'Size');
    freetri.feature('size1').set('custom', 'on');
    freetri.feature('size1').set('hmax', 'hmax_air');
    freetri.feature('size1').set('hmin', 'hmin');
%     freetri.feature('size1').set('custom', false);
%     freetri.feature('size1').set('hauto', 3);
    freetri.selection.set([m2a]);
    copy_face = mesh.create('cpf2a', 'CopyFace');
    copy_face.selection('source').set([m2a]);
    copy_face.selection('destination').set([m1a]);
    
    physics.feature.create('pc2', 'PeriodicCondition', 2);    % What does 2 stand for here?
    physics.feature('pc2').set('PeriodicType', 'Floquet');
    physics.feature('pc2').selection.set([m1 m1a m2 m2a]);
    
    rot = geometry.feature.create('rot3', 'Rotate');
    rot.set('rot', '60');
    rot.set('pos', {num2str(geom.mech.amech/2,'%10.9e') num2str(geom.mech.amech/sqrt(3),'%10.9e') '0'});
    rot.selection('input').set({'rot2'});
    
    mesh.create('ftet1', 'FreeTet');
    mesh.feature('ftet1').create('size1', 'Size');
    mesh.feature('ftet1').feature('size1').set('custom', 'on');
    mesh.feature('ftet1').feature('size1').set('hmax', 'hmax');
    mesh.feature('ftet1').feature('size1').set('hmin', 'hmin');
%     mesh.feature('ftet1').feature('size1').set('custom', false);
%     mesh.feature('ftet1').feature('size1').set('hauto', 3);
    mesh.feature('ftet1').selection.set(si_domain);
    mesh.feature('ftet1').feature('size1').selection.set(si_domain);
    
    mesh.create('ftet2', 'FreeTet');
    mesh.feature('ftet2').create('size1', 'Size');
    mesh.feature('ftet2').feature('size1').set('custom', 'on');
    mesh.feature('ftet2').feature('size1').set('hmax', 'hmax_air');
    mesh.feature('ftet2').feature('size1').set('hmin', 'hmin');
    mesh.feature('ftet2').selection.set(air_domain);
    mesh.feature('ftet2').feature('size1').selection.set(air_domain);
    
    coordBox = [-1, 1; -1, 1; 2*Eval_opt.lambda-epsilon, 2*Eval_opt.lambda+epsilon];
    sc1t = mphselectbox(model,'geom1',coordBox,'boundary');
    model.physics('emw').feature.create('sctr1', 'Scattering', 2);    % What does 2 stand for here?
    model.physics('emw').feature('sctr1').selection.set([sc1t]);

    mesh.run;
    figure(2)
    mphmesh(model)
    mphsave(model, Eval_opt.filename);

%%   COMSOL STUDY
  
    disp('Launching Study... ');
    study = model.study.create('std');
    studyEf = study.feature.create('eig', 'Eigenfrequency');
    studyEf.set('eigunit', 'Hz');
    studyEf.activate('emw', true);
    studyEf.set('neigs', num2str(Eval_opt.NumberOfFrequencies));
    studyEf.set('shift', num2str(Eval_opt.CentralFrequency));
%     studyEf.set('neigsactive', 'on');
%     studyEf.set('eigwhich', 'si');

    model.physics('emw').feature('pc1').set('kFloquet', {num2str(Eval_opt.kx,'%10.9e') num2str(Eval_opt.ky,'%10.9e') '0'});
    model.physics('emw').feature('pc2').set('kFloquet', {num2str(Eval_opt.kx,'%10.9e') num2str(Eval_opt.ky,'%10.9e') '0'});

    study.run;
    data_freq = mpheval(model,{'real(freq)'});
    data_damp = mpheval(model,{'imag(freq)'});
    
    coordBox = [-1, 1; -1, 1; 2*Eval_opt.lambda - epsilon, 2*Eval_opt.lambda + epsilon];
    sur_top = mphselectbox(model,'geom1',coordBox,'boundary');
    
    coordBox = [-1, 1; -1, 1; -epsilon, geom.mech.d/2 + epsilon];
    all_boundaries = mphselectbox(model,'geom1',coordBox, 'boundary');
    
    floquet_boundaries = [];
    floquet_boundaries = [floquet_boundaries; model.physics('emw').feature('pc1').selection().inputEntities];
    floquet_boundaries = [floquet_boundaries; model.physics('emw').feature('pc2').selection().inputEntities];
    
    sym_boundary = [];
    sym_boundary = [sym_boundary; model.physics('emw').feature('pmc1').selection().inputEntities];
    
    coordBox = [-1, 1; -1, 1; geom.mech.d/2 - epsilon, geom.mech.d/2 + epsilon];
    si_top = mphselectbox(model,'geom1',coordBox, 'boundary');
    
    floquet_Si = intersect(all_boundaries, floquet_boundaries);
    floquet_air = setdiff(floquet_boundaries, floquet_Si);
    
    sur_xy = setdiff(all_boundaries, sym_boundary);
    sur_xy = setdiff(sur_xy, floquet_Si);
    sur_xy = setdiff(sur_xy, sur_top);
    sur_xy = setdiff(sur_xy, si_top);
    sur_xy = horzcat(sur_xy, transpose(floquet_air));

    for j=1:min(size(Eval_opt.freq,1), size(data_freq.d1,1))
        Eval_opt.freq(j)=data_freq.d1(j,1);
        Eval_opt.damp(j)=data_damp.d1(j,1);
        
        Eval_opt.qual(j) = Eval_opt.freq(j)/(2*Eval_opt.damp(j));
        
        
        [Eval_opt.Eint(j), Eval_opt.unitEint] = mphint2(model,'emw.Wav','volume','solnum',j);
        
        [Eval_opt.flux_z(j), Eval_opt.unitflux_z] = mphint2(model, 'emw.nPoav',...
        'surface','selection',sur_top,'solnum',j);
        Eval_opt.flux_z(j) = Eval_opt.flux_z(j)/Eval_opt.Eint(j);

        [Eval_opt.flux_xy(j), Eval_opt.unitflux_xy] = mphint2(model, 'emw.nPoav',...
        'surface','selection',sur_xy,'solnum',j);
        Eval_opt.flux_xy(j) = Eval_opt.flux_xy(j)/Eval_opt.Eint(j);
        
        Eval_opt.coupling(j) = Eval_opt.flux_z(j)/(Eval_opt.flux_z(j) + Eval_opt.flux_xy(j));
    end
    
    [Eval_opt.quality, Eval_opt.index] = max(Eval_opt.qual);
    Eval_opt.frequency = Eval_opt.freq(Eval_opt.index);
    Eval_opt.damping = Eval_opt.damp(Eval_opt.index);
    
%% Plotting all the optical modes with Q>1e4

    index = find(real(Eval_opt.qual)>Eval_opt.qual_limit);

    model.result.create('pg2', 'PlotGroup3D');
    model.result('pg2').set('frametype', 'spatial');
    model.result('pg2').set('data', 'dset1');
    model.result('pg2').feature.create('mslc1', 'Multislice');
    model.result('pg2').feature('mslc1').label('Multislice');
    model.result('pg2').feature('mslc1').set('data', 'parent');
    model.result('pg2').feature('mslc1').set('expr', 'emw.Ex');
    model.result('pg2').feature('mslc1').set('descr', 'Electric field, x component');

    model.result('pg2').feature('mslc1').set('xnumber', '0');
    model.result('pg2').feature('mslc1').set('ynumber', '0');
    model.result('pg2').feature('mslc1').set('multiplanezmethod', 'coord');
    model.result('pg2').feature('mslc1').set('zcoord', '0');

    for j=1:length(index)
        temp = figure;
        clf;
        hold on;

        model.result('pg2').set('looplevel', {num2str(index(j))});
        mphplot(model, 'pg2');
        view(0,90)
        box on

        savefig(temp, strcat('index_', num2str(index(j)), '_wavelength_', num2str(c/Eval_opt.freq(index(j))), '_Qt_ ', num2str(Eval_opt.qual(index(j))), '_coupling_ ', num2str(Eval_opt.coupling(index(j))), '_flux_xy_', num2str(Eval_opt.flux_xy(index(j))), '_flux_z_', num2str(Eval_opt.flux_z(index(j))), '.fig'))
    end

    save(Eval_opt.filename, 'Eval_opt', 'geom');
    mphsave(model,Eval_opt.filename);
    
    Eval_opt.time = toc;
    
    save(Eval_opt.filename, 'Eval_opt', 'geom');
%%     
end